----------------------------------------------------------------------------------
-- Company: TalTech
-- Engineer: Artjom Jukov
-- 
-- Create Date: 04/16/2019 08:20:34 PM
-- Design Name: 
-- Module Name: main - Behavioral
-- Project Name: IAPB
-- Target Devices: Basys 3 
-- Tool Versions: Vivado 2018.3
-- Description: 
-- Please read LICENSE terms at 
-- https://gitlab.cs.ttu.ee/arjusk/iapb
-- Dependencies: 
-- 
-- Revision:
-- Revision 0.01 - File Created
-- Additional Comments:
-- Copyright (c) 2019 Artjom Jukov

----------------------------------------------------------------------------------
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;

-- Uncomment the following library declaration if using
-- arithmetic functions with Signed or Unsigned values
use IEEE.NUMERIC_STD.ALL;

-- Uncomment the following library declaration if instantiating
-- any Xilinx leaf cells in this code.
--library UNISIM;
--use UNISIM.VComponents.all;

entity main is
    Port ( 
           sw : in std_logic_vector(15 downto 0);
           clk : in std_logic;
           btnr, btnu, btnd, btnc, btnl  : in std_logic;
           seg : out std_logic_vector(6 downto 0);
           led : out std_logic_vector(15 downto 0);
           an : out std_logic_vector(3 downto 0)
           );
end main;

architecture Behavioral of main is

    --Display entity component
    component seven_segment_display
    port(
        clk : in STD_LOGIC;
        displayed_number: in STD_LOGIC_VECTOR(15 downto 0):= (others => '0');
        an : out STD_LOGIC_VECTOR(3 downto 0);
        seg : out STD_LOGIC_VECTOR(6 downto 0)
    );
    end component;
    
    --Debounce entity component
    component debounce_button
      port (
        clk    : in  std_logic;
        in_button_signal : in  std_logic;
        out_debounced_signal : out std_logic
        );
    end component;
    
    --GCD example algorithm component
    component example_code is 
      port (xi, yi : in unsigned(15 downto 0);
            rst    : in std_logic;
            sel    : in std_logic_vector(1 downto 0);
            xo, dbg: out unsigned(15 downto 0);
            rdy    : out std_logic;
            clk    : in std_logic);
    end component;
    
    --Bachelor algorithm component
    component algorithm is
    port (    clk, rst: in std_logic;
              rdy: out std_logic;
             a, b: in unsigned(0 to 7);
            c, c2: out unsigned(0 to 15) := (others=>'0');
              sel: in std_logic_vector(0 to 1);
	          dbg: out unsigned(0 to 15)
                );
    end component;
    
    --Decalrations
    type state_type is (s_start, s_manual, s_reset1, s_reset2, s_run, s_ready);
    signal state, next_state: state_type;
    
    signal first_number, second_number, displayed_number: 
                                 std_logic_vector(15 downto 0):= (others => '0');
    signal debounced_btnu, debounced_btnr, debounced_btnd, rdy: std_logic := '0';
    constant BTN_ACTIVE : std_logic := '1';
    signal cclk, rst1, manual, algoritm_reset : std_logic;
    signal sel: unsigned(1 downto 0) := "00";
    signal dbg, result1, result2 : std_logic_vector(15 downto 0);
    
begin
    
    --Register
    --Switches signals saving to registers
    process(btnc, btnl, clk)
    begin 
        if rising_edge(clk) then
            --When left button pushed, 
            --saving switches signals to internal register
            if btnl = BTN_ACTIVE then
                first_number <= sw;
            --When center button pushed, 
            --saving switches signals to internal register
            elsif btnc = BTN_ACTIVE then
                second_number <= sw;
            end if;
         end if;
    end process;
       

    --Multiplexor
    --The process which decides what register will be displayed
    process(sw, dbg, result1, result2, first_number, second_number, btnl, btnc)
    begin
        --When left button pushed,
        --on display will be shown intrenal register
        --with first number
        if btnl = BTN_ACTIVE then
           displayed_number <= first_number;
           
        --When center button pushed,
        --on display will be shown intrenal register
        --with second number
        elsif btnc = BTN_ACTIVE then
           displayed_number <= second_number;
        else 
            --Debug mode for algorithm
            --on display will be shown algorithm entity 
            --internal registers 
            if sw(15) = '1' then 
                displayed_number <= dbg;
            else 
                --If debug mode is off on display will be shown 
                --main entity internal registers, 
                --depends on sw14 and sw13 signals
                case sw(14 downto 13) is 
                    when "00" => displayed_number <= first_number;
                    when "01" => displayed_number <= second_number;
                    when "10" => displayed_number <= result1;
                    when "11" => displayed_number <= result2;
                    when others => displayed_number <=(others => '0');
                end case;
            end if;   
        end if;
    end process;   
    
    --Display entity ports mapping
    sev_segment : seven_segment_display
    port map (
        clk => clk,
        displayed_number => displayed_number,
        an => an,
        seg => seg
    );
    
    --Debounce signal of right button
    debouncer_btnr : debounce_button
    port map(
        clk => clk,
        in_button_signal => btnr,
        out_debounced_signal=> debounced_btnr
    );
    
    --Debounce signal of upper button
    debounce_btnu : debounce_button
    port map(
        clk => clk,
        in_button_signal => btnu,
        out_debounced_signal=> debounced_btnu
    );   
    
    --Debounce signal of down button
    debounce_btnd : debounce_button
    port map(
        clk => clk,
        in_button_signal => btnd,
        out_debounced_signal=> debounced_btnd
    ); 
    
    --Holding algorithm in manual mode
    --Control machine (Moore) which changes state based on some arguments
    --controls automatic algorithm run, so algorithm is running only once.
    process(debounced_btnd, debounced_btnu)
    begin
        --Default values
        rst1 <= '0';
        manual <= '0';
        next_state <= state;
        led(15 downto 10) <= "000000";
        --If two button pushed, start changing state
        if debounced_btnd = '1'  and debounced_btnu = '1' then 
            next_state <= s_start;
            manual <= '1';
        else 
            case state is 
                --Start state
                -- When user release upper button going to manual state 
                when s_start =>
                    led (15 downto 10) <= "100000"; 
                    manual <= '1';
                    if debounced_btnu = '0' then
                        next_state <= s_manual;
                    end if;
                --Manual state
                --When upper button is pushed going to reset state
                when s_manual =>
                    led (15 downto 10) <= "010000";
                    manual <= '1';
                    if debounced_btnu = '1' then
                        next_state <= s_reset1;
                    end if;
                --Second reset state to be sure, first reset is button
                when s_reset1 =>
                    led (15 downto 10) <= "001000";
                    rst1 <= '1';
                    next_state <= s_reset2;
                --Third reset state to be sure
                when s_reset2 =>
                    led (15 downto 10) <= "000100";
                    rst1 <= '1'; 
                    next_state <= s_run;
                --Run state
                --Waiting until algorith signals that it's finished 
                --(ready signal from algorith)
                --when ready signal is '1' going to ready state
                when s_run =>
                    led (15 downto 10) <= "000010";
                    if rdy = '1' then
                        next_state <= s_ready;
                    end if;
                -- Ready state
                -- Waiting until reset button (down button) is pushed, 
                --then algorithm setting ready signal to '0'
                when s_ready =>
                    led (15 downto 10) <= "000001";
                    if rdy = '0' then
                        next_state <= s_start;
                    end if;
            end case;
        end if;
    end process; 
        
    --Register
    process(clk, debounced_btnr, manual)
    begin
        --When run (upper button) didn't push
        --then set cclk debounced right button signal,
        --this is nessecary, to stop algorithm running through
        if manual = '1' then 
            cclk <= debounced_btnr;
        --When run (upper button) had pushed
        --then clk will be set to cclk (main internal register),
        --which will be given to algorithm 
        else
            cclk <= clk;
        end if;        
    end process;
   
  --Register
  --when clk is '1' set to internal register
  --buffer register value
  process(clk)
  begin
    if rising_edge(clk) then
        state <= next_state;
    end if;
  end process;
    
    
    --Teacher algorithm of GCD
    algorithm_GCD : example_code
    port map (
        xi => unsigned(first_number),
        yi => unsigned(second_number),
        rst => debounced_btnd,
        sel => sw(14 downto 13),
        std_logic_vector(xo) => result1,
        std_logic_vector(dbg) => dbg,
        rdy => rdy,
        clk => cclk
    );
    
    --Assigne to internal register button signal or reset from control machine
    -- one of them need to be 1, so algorithm will be reset
    algoritm_reset <= debounced_btnd or rst1;
    
    --Bachelor algorithm rebuild.
--    examp_algorithm : algorithm
--    port map( clk => cclk,
--              a => unsigned(first_number(7 downto 0)),  
--              b => unsigned(second_number(7 downto 0)),
--             std_logic_vector(c) => result1,
--             std_logic_vector(c2) => result2,
--             sel => sw(14 downto 13),
--             std_logic_vector(dbg) => dbg,
--             rst => algoritm_reset,
--             rdy => rdy
--     );
     
     --Assigne to led indicators internal registers signals
     led(3) <= cclk;   
     led(2) <= algoritm_reset;
     led(1) <= rdy;
     led(0) <= manual;

    
end Behavioral;
        